Translating RNA and protein sequences
=====================================


An example for how to parse a sequence from a String and using the Translation engine to convert into amino acid sequence. 

```java    
  String dnaFastaS = ">gb:GQ903697|Organism:Arenavirus H0030026 H0030026|Segment:S|Host:Rat\n" +
                "CGCACAGAGGATCCTAGGCGTTACTGACTTGCGCTAATAACAGATACTGTTTCATATTTAGATAAAGACC\n" +
                "CAGCCAACTGATTGGTCAGCATGGGACAACTTGTGTCCCTCTTCAGTGAAATTCCATCAATCATACACGA\n" +
                "AGCTCTCAATGTTGCTCTCGTAGCTGTTAGCATCATTGCAATATTGAAAGGGGTTGTGAATGTTTGGAAG\n" +
                "AGTGGAGTTTTGCAGCTTTTGGCCTTCTTGCTCCTGGCGGGAAGATCCTGCTCAGTCATAATTGGTCATC\n" +
                "ATCTCGAACTGCAGCATGTGATCTTCAATGGGTCATCAATCACACCCTTTTTACCAGTTACATGTAAGAT\n" +
                "CAATGATACCTACTTCCTACTAAGAGGCCCCTATGAAGCTGATTGGGCAGTTGAATTGAGTGTAACTGAA\n" +
                "ACCACAGTCTTGGTTGATCTTGAAGGTGGCAGCTCAATGAAGCTGAAAGCCGGAAACATCTCAGGTTGTC\n" +
                "TTGGAGACAACCCCCATCTGAGATCAGTGGTCTTCACATTGAATTGGTTGCTAACAGGATTAGATCATGT\n" +
                "TATTGATTCTGACCCGAAAATTCTCTGTGATCTTAAAGACAGTGGGCACTTTCGTCTCCAGATGAACTTA\n" +
                "ACAGAAAAGCACTATTGTGACAAGTTTCACATCAAAATGGGCAAGGTCTTTGGCGTATTCAAAGATCCGT\n" +
                "GCATGGCTGGTGGTAAAATGTTTGCCATACTAAAAAATACCTCTTGGTCGAACCAGTGCCAAGGAAACCA\n" +
                "TGTCAGCACCATTCATCTTGTCCTTCAGAGTAATTTCAAACAGGTCCTCAGTAGCAGGAAACTGTTGAAC\n" +
                "TTTTTCAGCTGGTCATTGTCTGATGCCACAGGGGCTGATATGCCTGGTGGTTTTTGTCTGGAAAAATGGA\n" +
                "TGTTGATTTCAAGTGAACTGAAATGCTTTGGAAACACAGCTGTGGCAAAGTGCAACTTAAATCATGACTC\n" +
                "AGAGTTCTGTGACATGCTTAGGCTTTTTGATTTCAACAAAAAGGCAATAGTCACTCTTCAGAACAAAACA\n" +
                "AAGCATCGGCTGGACACAGTAATTACTGCTATCAATTCATTGATCTCTGATAATATTCTTATGAAGAACA\n" +
                "GGATTAAAGAATTGATAGATGTTCCTTACTGTAATTACACCAAATTTTGGTATGTCAATCACACAGGTCT\n" +
                "AAATCTGCACACCCTTCCAAGATGTTGGCTTGTTAAAAATGGTAGCTACTTGAATGTGTCTGACTTCAGG\n" +
                "AATGAGTGGATATTGGAGAGTGATCATCTTGTTTCGGAGATCCTTTCAAAGGAGTATGAGGAAAGGCAAA\n" +
                "ATCGTACACCACTCTCACTGGTTGACATCTGTTTCTGGAGTACATTGTTTTACACAGCATCAATTTTCCT\n" +
                "ACACCTCTTGAGAATTCCAACCCACAGACACATTGTTGGTGAGGGCTGCCCGAAGCCTCATAGGCTAAAC\n" +
                "AGGCACTCAATATGTGCTTGTGGCCTTTTCAAACAAGAAGGCAGACCCTTGAGATGGGTAAGAAAGGTGT\n" +
                "GAACAATGGTTGCTTGGTGGCCTCCATTGCTGCACCCCCCTAGGGGGGTGCAGCAATGGAGGTTCTCGYT\n" +
                "GAGCCTAGAGAACAACTGTTGAATCGGGTTCTCTAAAGAGAACATCGATTGGTAGTACCCTTTTTGGTTT\n" +
                "TTCATTGGTCACTGACCCTGAAAGCACAGCACTGAACATCAAACAGTCCAAAAGTGCACAGTGTGCATTT\n" +
                "GTTGTGGCTGGTGCTGATCCTTTCTTCTTACTTTTAATGACTATTCCCTTATGTCTGTCACACAGATGTT\n" +
                "CAAATCTCTTCCAAACAAGATCTTCAAAGAGCCGTGACTGTTCTGCGGTCAGTTTGACATCAACAATCTT\n" +
                "CAAATCCTGTCTTCCATGCATATCAAAGAGCCTCCTAATATCATCAGCACCTTGCGCAGTGAAAACCATG\n" +
                "GATTTAGGCAGACTCCTTATTATGCTTGTGATGAGGCCAGGTCGTGCATGTTCAACATCCTTCAGCAATA\n" +
                "TCCCATGACAATATTTACTTTGGTCCTTAAAAGATTTTATGTCATTGGGTTTTCTGTAGCAGTGGATGAA\n" +
                "TTTTTGTGATTCAGGCTGGTAAATTGCAAACTCAACAGGGTCATGTGGCGGGCCTTCAATGTCAATCCAT\n" +
                "GTTGTGTCACTGACCATCAACGACTCTACACTTCTCTTCACCTGAGCCTCCACCTCAGGCTTGAGCGTGG\n" +
                "ACAAGAGTGGGGCACCACCGTTCCGGATGGGGACTGGTGTTTTGCTTGGTAAACTCTCAAATTCCACAAC\n" +
                "TGTATTGTCCCATGCTCTCCCTTTGATCTGTGATCTTGATGAAATGTAAGGCCAGCCCTCACCAGAGAGA\n" +
                "CACACCTTATAAAGTATGTTTTCATAAGGATTCCTCTGTCCTGGTATGGCACTGATGAACATGTTTTCCC\n" +
                "TCTTTTTGATCTCCAAGAGGGTTTTTATAATGGTTGTGAATGTGGACTCCTCAATCTTTATTGTTTCCAG\n" +
                "CATGTTGCCACCATCAATCAGGCAAGCACCGGCTTTCACAGCAGCTGATAAACTAAGGTTGTAGCCTGAT\n" +
                "ATGTTAATTTGAGAATCCTCCTGAGTGATTACCTTTAGAGAAGGATGCTTCTCCATCAAAGCATCTAAGT\n" +
                "CACTTAAATTAGGGTATTTTGCTGTGTATAGCAACCCCAGATCTGTGAGGGCCTGAACCACATCATTTAG\n" +
                "AGTTTCCCCTCCCTGTTCAGTCATACAGGAAATTGTGAGTGCTGGCATCGATCCAAATTGGTTGATCATA\n" +
                "AGTGATGAGTCTTTAACGTCCCAGACTTTGACCACCCCTCCAGTTCTAGCCAACCCAGGTCTCTGAATAC\n" +
                "CAACAAGTTGCAGAATTTCGGACCTCCTGGTGAGCTGTGTTGTAGAGAGGTTCCCTAGATACTGGCCACC\n" +
                "TGTGGCTGTCAACCTCTCTGTTCTTTGAACTTTTTGCCTTAATTTGTCCAAGTCACTGGAGAGTTCCATT\n" +
                "AGCTCTTCCTTTGACAATGATCCTATCTTAAGGAACATGTTCTTTTGGGTTGACTTCATGACCATCAATG\n" +
                "AGTCAACTTCCTTATTCAAGTCCCTCAAACTAACAAGATCACTGTCATCTCTTTTAGACCTCCTCATCAT\n" +
                "GCGTTGCACACTTGCAACCTTTGAAAAATCTAAGCCGGACAGAAGAGCCCTCGCGTCAGTTAGGACATCT\n" +
                "GCCTTAACAGCAGTTGTCCAGTTCGAGAGTCCTCTCCTGAGAGACTGTGTCCATCTGAATGATGGGATTG\n" +
                "GTTGTTCGCTCATAGTGATGAAATTGCGCAGAGTTATCCAAAAGCCTAGGATCCTCTGTGCG";


            try {

                // parse the raw sequence from the string
                InputStream stream = new ByteArrayInputStream(dnaFastaS.getBytes());

                // define the Ambiguity Compound Sets
                AmbiguityDNACompoundSet ambiguityDNACompoundSet = AmbiguityDNACompoundSet.getDNACompoundSet();
                CompoundSet<NucleotideCompound>  nucleotideCompoundSet = AmbiguityRNACompoundSet.getDNACompoundSet();

                FastaReader<DNASequence, NucleotideCompound> proxy =
                        new FastaReader<DNASequence, NucleotideCompound>(
                                stream,
                                new GenericFastaHeaderParser<DNASequence, NucleotideCompound>(),
                                new DNASequenceCreator(ambiguityDNACompoundSet));

                // has only one entry in this example, but could be easily extended to parse a FASTA file with multiple sequences
                LinkedHashMap<String, DNASequence> dnaSequences = proxy.process();

                // Initialize the Transcription Engine
                TranscriptionEngine engine = new
                        TranscriptionEngine.Builder().dnaCompounds(ambiguityDNACompoundSet).rnaCompounds(nucleotideCompoundSet).build();

                Frame[] sixFrames = Frame.getAllFrames();

                for (DNASequence dna : dnaSequences.values()) {

                    Map<Frame, Sequence<AminoAcidCompound>> results = engine.multipleFrameTranslation(dna, sixFrames);

                    for (Frame frame : sixFrames){
                        System.out.println("Translated Frame:" + frame +" : " + results.get(frame));
                    }

                }
            } catch (Exception e){
                e.printStackTrace();
        }
```  

This code will print out:

```
Translated Frame:ONE : RTEDPRRY*LALITDTVSYLDKDPAN*LVSMGQLVSLFSEIPSIIHEALNVALVAVSIIAILKGVVNVWKSGVLQLLAFLLLAGRSCSVIIGHHLELQHVIFNGSSITPFLPVTCKINDTYFLLRGPYEADWAVELSVTETTVLVDLEGGSSMKLKAGNISGCLGDNPHLRSVVFTLNWLLTGLDHVIDSDPKILCDLKDSGHFRLQMNLTEKHYCDKFHIKMGKVFGVFKDPCMAGGKMFAILKNTSWSNQCQGNHVSTIHLVLQSNFKQVLSSRKLLNFFSWSLSDATGADMPGGFCLEKWMLISSELKCFGNTAVAKCNLNHDSEFCDMLRLFDFNKKAIVTLQNKTKHRLDTVITAINSLISDNILMKNRIKELIDVPYCNYTKFWYVNHTGLNLHTLPRCWLVKNGSYLNVSDFRNEWILESDHLVSEILSKEYEERQNRTPLSLVDICFWSTLFYTASIFLHLLRIPTHRHIVGEGCPKPHRLNRHSICACGLFKQEGRPLRWVRKV*TMVAWWPPLLHPPRGVQQWRFSXSLENNC*IGFSKENIDW*YPFWFFIGH*P*KHSTEHQTVQKCTVCICCGWC*SFLLTFNDYSLMSVTQMFKSLPNKIFKEP*LFCGQFDINNLQILSSMHIKEPPNIISTLRSENHGFRQTPYYACDEARSCMFNILQQYPMTIFTLVLKRFYVIGFSVAVDEFL*FRLVNCKLNRVMWRAFNVNPCCVTDHQRLYTSLHLSLHLRLERGQEWGTTVPDGDWCFAW*TLKFHNCIVPCSPFDL*S**NVRPALTRETHLIKYVFIRIPLSWYGTDEHVFPLFDLQEGFYNGCECGLLNLYCFQHVATINQASTGFHSS**TKVVA*YVNLRILLSDYL*RRMLLHQSI*VT*IRVFCCV*QPQICEGLNHII*SFPSLFSHTGNCECWHRSKLVDHK**VFNVPDFDHPSSSSQPRSLNTNKLQNFGPPGELCCREVP*ILATCGCQPLCSLNFLP*FVQVTGEFH*LFL*Q*SYLKEHVLLG*LHDHQ*VNFLIQVPQTNKITVISFRPPHHALHTCNL*KI*AGQKSPRVS*DICLNSSCPVRESSPERLCPSE*WDWLFAHSDEIAQSYPKA*DPLC
Translated Frame:TWO : AQRILGVTDLR**QILFHI*IKTQPTDWSAWDNLCPSSVKFHQSYTKLSMLLS*LLASLQY*KGL*MFGRVEFCSFWPSCSWREDPAQS*LVIISNCSM*SSMGHQSHPFYQLHVRSMIPTSY*EAPMKLIGQLN*V*LKPQSWLILKVAAQ*S*KPETSQVVLETTPI*DQWSSH*IGC*QD*IMLLILTRKFSVILKTVGTFVSR*T*QKSTIVTSFTSKWARSLAYSKIRAWLVVKCLPY*KIPLGRTSAKETMSAPFILSFRVISNRSSVAGNC*TFSAGHCLMPQGLICLVVFVWKNGC*FQVN*NALETQLWQSAT*IMTQSSVTCLGFLISTKRQ*SLFRTKQSIGWTQ*LLLSIH*SLIIFL*RTGLKN**MFLTVITPNFGMSITQV*ICTPFQDVGLLKMVAT*MCLTSGMSGYWRVIILFRRSFQRSMRKGKIVHHSHWLTSVSGVHCFTQHQFSYTS*EFQPTDTLLVRAARSLIG*TGTQYVLVAFSNKKADP*DG*ERCEQWLLGGLHCCTPLGGCSNGGSX*A*RTTVESGSLKRTSIGSTLFGFSLVTDPESTALNIKQSKSAQCAFVVAGADPFFLLLMTIPLCLSHRCSNLFQTRSSKSRDCSAVSLTSTIFKSCLPCISKSLLISSAPCAVKTMDLGRLLIMLVMRPGRACSTSFSNIP*QYLLWSLKDFMSLGFL*QWMNFCDSGW*IANSTGSCGGPSMSIHVVSLTINDSTLLFT*ASTSGLSVDKSGAPPFRMGTGVLLGKLSNSTTVLSHALPLICDLDEM*GQPSPERHTL*SMFS*GFLCPGMALMNMFSLFLISKRVFIMVVNVDSSIFIVSSMLPPSIRQAPAFTAADKLRL*PDMLI*ESS*VITFREGCFSIKASKSLKLGYFAVYSNPRSVRA*TTSFRVSPPCSVIQEIVSAGIDPNWLIISDESLTSQTLTTPPVLANPGL*IPTSCRISDLLVSCVVERFPRYWPPVAVNLSVL*TFCLNLSKSLESSISSSFDNDPILRNMFFWVDFMTINESTSLFKSLKLTRSLSSLLDLLIMRCTLATFEKSKPDRRALASVRTSALTAVVQFESPLLRDCVHLNDGIGCSLIVMKLRRVIQKPRILCA
Translated Frame:THREE : HRGS*ALLTCANNRYCFIFR*RPSQLIGQHGTTCVPLQ*NSINHTRSSQCCSRSC*HHCNIERGCECLEEWSFAAFGLLAPGGKILLSHNWSSSRTAACDLQWVINHTLFTSYM*DQ*YLLPTKRPL*S*LGS*IECN*NHSLG*S*RWQLNEAESRKHLRLSWRQPPSEISGLHIELVANRIRSCY*F*PENSL*S*RQWALSSPDELNRKALL*QVSHQNGQGLWRIQRSVHGWW*NVCHTKKYLLVEPVPRKPCQHHSSCPSE*FQTGPQ*QETVELFQLVIV*CHRG*YAWWFLSGKMDVDFK*TEMLWKHSCGKVQLKS*LRVL*HA*AF*FQQKGNSHSSEQNKASAGHSNYCYQFIDL**YSYEEQD*RIDRCSLL*LHQILVCQSHRSKSAHPSKMLAC*KW*LLECV*LQE*VDIGE*SSCFGDPFKGV*GKAKSYTTLTG*HLFLEYIVLHSINFPTPLENSNPQTHCW*GLPEAS*AKQALNMCLWPFQTRRQTLEMGKKGVNNGCLVASIAAPP*GGAAMEVLXEPREQLLNRVL*REHRLVVPFLVFHWSLTLKAQH*TSNSPKVHSVHLLWLVLILSSYF**LFPYVCHTDVQISSKQDLQRAVTVLRSV*HQQSSNPVFHAYQRAS*YHQHLAQ*KPWI*ADSLLCL**GQVVHVQHPSAISHDNIYFGP*KILCHWVFCSSG*IFVIQAGKLQTQQGHVAGLQCQSMLCH*PSTTLHFSSPEPPPQA*AWTRVGHHRSGWGLVFCLVNSQIPQLYCPMLSL*SVILMKCKASPHQRDTPYKVCFHKDSSVLVWH**TCFPSF*SPRGFL*WL*MWTPQSLLFPACCHHQSGKHRLSQQLIN*GCSLIC*FENPPE*LPLEKDASPSKHLSHLN*GILLCIATPDL*GPEPHHLEFPLPVQSYRKL*VLASIQIG*S*VMSL*RPRL*PPLQF*PTQVSEYQQVAEFRTSW*AVL*RGSLDTGHLWLSTSLFFELFALICPSHWRVPLALPLTMILS*GTCSFGLTS*PSMSQLPYSSPSN*QDHCHLF*TSSSCVAHLQPLKNLSRTEEPSRQLGHLP*QQLSSSRVLS*ETVSI*MMGLVVRS***NCAELSKSLGSSV
Translated Frame:REVERSED_ONE : RTEDPRLLDNSAQFHHYERTTNPIIQMDTVSQERTLELDNCC*GRCPN*REGSSVRLRFFKGCKCATHDEEV*KR*Q*SC*FEGLE*GS*LIDGHEVNPKEHVP*DRIIVKGRANGTLQ*LGQIKAKSSKNREVDSHRWPVSREPLYNTAHQEVRNSATCWYSETWVG*NWRGGQSLGR*RLITYDQPIWIDASTHNFLYD*TGRGNSK*CGSGPHRSGVAIHSKIP*FK*LRCFDGEASFSKGNHSGGFSN*HIRLQP*FISCCESRCLPD*WWQHAGNNKD*GVHIHNHYKNPLGDQKEGKHVHQCHTRTEESL*KHTL*GVSLW*GLALHFIKITDQRESMGQYSCGI*EFTKQNTSPHPERWCPTLVHAQA*GGGSGEEKCRVVDGQ*HNMD*H*RPAT*PC*VCNLPA*ITKIHPLLQKTQ*HKIF*GPK*ILSWDIAEGC*TCTTWPHHKHNKESA*IHGFHCARC**Y*EAL*YAWKTGFEDC*CQTDRRTVTAL*RSCLEEI*TSV*QT*GNSH*K*EERISTSHNKCTLCTFGLFDVQCCAFRVSDQ*KTKKGTTNRCSL*RTRFNSCSLGSXRTSIAAPP*GGAAMEATKQPLFTPFLPISRVCLLV*KGHKHILSACLAYEASGSPHQQCVCGLEFSRGVGKLMLCKTMYSRNRCQPVRVVYDFAFPHTPLKGSPKQDDHSPISTHS*SQTHSSSYHF*QANILEGCADLDLCD*HTKIWCNYSKEHLSIL*SCSS*EYYQRSMN**Q*LLCPADALFCSEE*LLPFC*NQKA*ACHRTLSHDLSCTLPQLCFQSISVHLKSTSIFPDKNHQAYQPLWHQTMTS*KSSTVSCY*GPV*NYSEGQDEWC*HGFLGTGSTKRYFLVWQTFYHQPCTDL*IRQRPCPF*CETCHNSAFLLSSSGDESAHCL*DHREFSGQNQ*HDLILLATNSM*RPLISDGGCLQDNLRCFRLSASLSCHLQDQPRLWFQLHSIQLPNQLHRGLLVGSRYH*SYM*LVKRV*LMTH*RSHAAVRDDDQL*LSRIFPPGARRPKAAKLHSSKHSQPLSILQ*C*QLREQH*ELRV*LMEFH*RGTQVVPC*PISWLGLYLNMKQYLLLAQVSNA*DPLC
Translated Frame:REVERSED_TWO : AQRILGFWITLRNFITMSEQPIPSFRWTQSLRRGLSNWTTAVKADVLTDARALLSGLDFSKVASVQRMMRRSKRDDSDLVSLRDLNKEVDSLMVMKSTQKNMFLKIGSLSKEELMELSSDLDKLRQKVQRTERLTATGGQYLGNLSTTQLTRRSEILQLVGIQRPGLARTGGVVKVWDVKDSSLMINQFGSMPALTISCMTEQGGETLNDVVQALTDLGLLYTAKYPNLSDLDALMEKHPSLKVITQEDSQINISGYNLSLSAAVKAGACLIDGGNMLETIKIEESTFTTIIKTLLEIKKRENMFISAIPGQRNPYENILYKVCLSGEGWPYISSRSQIKGRAWDNTVVEFESLPSKTPVPIRNGGAPLLSTLKPEVEAQVKRSVESLMVSDTTWIDIEGPPHDPVEFAIYQPESQKFIHCYRKPNDIKSFKDQSKYCHGILLKDVEHARPGLITSIIRSLPKSMVFTAQGADDIRRLFDMHGRQDLKIVDVKLTAEQSRLFEDLVWKRFEHLCDRHKGIVIKSKKKGSAPATTNAHCALLDCLMFSAVLSGSVTNEKPKRVLPIDVLFREPDSTVVL*AXREPPLLHPPRGVQQWRPPSNHCSHLSYPSQGSAFLFEKATSTY*VPV*PMRLRAALTNNVSVGWNSQEV*EN*CCVKQCTPETDVNQ*EWCTILPFLILL*KDLRNKMITLQYPLIPEVRHIQVATIFNKPTSWKGVQI*TCVIDIPKFGVITVRNIYQFFNPVLHKNIIRDQ*IDSSNYCVQPMLCFVLKSDYCLFVEIKKPKHVTEL*VMI*VALCHSCVSKAFQFT*NQHPFFQTKTTRHISPCGIRQ*PAEKVQQFPATEDLFEITLKDKMNGADMVSLALVRPRGIF*YGKHFTTSHARIFEYAKDLAHFDVKLVTIVLFC*VHLETKVPTVFKITENFRVRINNMI*SC*QPIQCEDH*SQMGVVSKTT*DVSGFQLH*AATFKINQDCGFSYTQFNCPISFIGAS**EVGIIDLTCNW*KGCD**PIEDHMLQFEMMTNYD*AGSSRQEQEGQKLQNSTLPNIHNPFQYCNDANSYESNIESFVYD*WNFTEEGHKLSHADQSVGWVFI*I*NSICY*RKSVTPRILCA
Translated Frame:REVERSED_THREE : HRGS*AFG*LCAISSL*ANNQSHHSDGHSLSGEDSRTGQLLLRQMS*LTRGLFCPA*IFQRLQVCNA**GGLKEMTVILLV*GT*IRKLTH*WS*SQPKRTCSLR*DHCQRKS*WNSPVTWTN*GKKFKEQRG*QPQVASI*GTSLQHSSPGGPKFCNLLVFRDLGWLELEGWSKSGTLKTHHL*STNLDRCQHSQFPV*LNREGKL*MMWFRPSQIWGCYTQQNTLI*VT*ML*WRSILL*R*SLRRILKLTYQATTLVYQLL*KPVLA*LMVATCWKQ*RLRSPHSQPL*KPSWRSKRGKTCSSVPYQDRGILMKTYFIRCVSLVRAGLTFHQDHRSKGEHGTIQLWNLRVYQAKHQSPSGTVVPHSCPRSSLRWRLR*REV*SR*WSVTQHGLTLKARHMTLLSLQFTSLNHKNSSTATENPMT*NLLRTKVNIVMGYC*RMLNMHDLASSQA**GVCLNPWFSLRKVLMILGGSLICMEDRI*RLLMSN*PQNSHGSLKILFGRDLNICVTDIRE*SLKVRRKDQHQPQQMHTVHFWTV*CSVLCFQGQ*PMKNQKGYYQSMFSLENPIQQLFSRLXENLHCCTPLGGCSNGGHQATIVHTFLTHLKGLPSCLKRPQAHIECLFSL*GFGQPSPTMCLWVGILKRCRKIDAV*NNVLQKQMSTSESGVRFCLSSYSFERISETR*SLSNIHSFLKSDTFK*LPFLTSQHLGRVCRFRPV*LTYQNLV*LQ*GTSINSLILFFIRILSEINELIAVITVSSRCFVLF*RVTIAFLLKSKSLSMSQNSES*FKLHFATAVFPKHFSSLEINIHFSRQKPPGISAPVASDNDQLKKFNSFLLLRTCLKLL*RTR*MVLTWFPWHWFDQEVFFSMANILPPAMHGSLNTPKTLPILM*NLSQ*CFSVKFIWRRKCPLSLRSQRIFGSESIT*SNPVSNQFNVKTTDLRWGLSPRQPEMFPAFSFIELPPSRSTKTVVSVTLNSTAQSAS*GPLSRK*VSLILHVTGKKGVIDDPLKITCCSSR**PIMTEQDLPARSKKAKSCKTPLFQTFTTPFNIAMMLTATRATLRASCMIDGISLKRDTSCPMLTNQLAGSLSKYETVSVISASQ*RLGSSV
``` 
<!--automatically generated footer-->

---

Navigation:
[Home](../README.md)
| [Book 1: The Core module](README.md)
| Chapter 4 : Translating

Prev: [Chapter 2 : Basic Sequence types](sequences.md)
